package main

import (
	"log"

	"github.com/spf13/cobra"
	"gitlab.xaotos.cn/qtt/acmin/internal/config"
	"gitlab.xaotos.cn/qtt/acmin/internal/modules/context"
	"gitlab.xaotos.cn/qtt/acmin/pkg/cache"
	"gitlab.xaotos.cn/qtt/acmin/pkg/gormx"
	"gitlab.xaotos.cn/qtt/acmin/pkg/util/env"
	flagutil "gitlab.xaotos.cn/qtt/acmin/pkg/util/flag"
	"gorm.io/gorm"
)

const CommonPrefix = "acmin-"

var (
	configFile *string
	cacher     func() (*cache.Cache, error)
	dber       func() (*gorm.DB, error)
)

// config override order:
//
// env => config.yaml => flags default
func command() *cobra.Command {
	cmd := &cobra.Command{
		Run: func(cmd *cobra.Command, args []string) {
			opts := flagutil.Options{FlagPrefix: CommonPrefix}
			conf := &config.AcminConfig{
				Redis: config.RedisConfig{
					Addr: flagutil.GetFlagVal(cmd, opts, cache.FlagRedis, cmd.Flags().GetString)(),
					DB:   flagutil.GetFlagVal(cmd, opts, cache.FlagRedisDB, cmd.Flags().GetInt)(),
				},
				Database: config.DatabaseConfig{
					Driver:      flagutil.GetFlagVal(cmd, opts, gormx.FlagDriver, cmd.Flags().GetString)(),
					DSN:         flagutil.GetFlagVal(cmd, opts, gormx.FlagDsn, cmd.Flags().GetString)(),
					TablePrefix: flagutil.GetFlagVal(cmd, opts, gormx.FlagTablePrefix, cmd.Flags().GetString)(),
				},
			}
			// create app context
			appCtx := context.New(cacher, dber, conf)
			_ = appCtx
		},
	}

	cmd.Flags().StringVarP(configFile, "config", "c", env.StringFromEnv("ACMIN_CONFIG", "configs/config.yaml"), "")
	cacher = cache.AddRedisCacheFlagsToCmd(cmd, cache.Options{FlagPrefix: CommonPrefix})
	dber = gormx.AddGormDBFlagsToCmd(cmd, gormx.Options{FlagPrefix: CommonPrefix})
	return cmd
}

func main() {
	cobra.OnInitialize(config.InitFunc(configFile))
	log.Fatal((command().Execute()))
}
