package cache

import (
	"context"
	"errors"
	"fmt"
	"math"
	"os"
	"strings"
	"time"

	"github.com/redis/go-redis/v9"
	"github.com/spf13/cobra"
	"gitlab.xaotos.cn/qtt/acmin/pkg/util/env"
)

var (
	ErrCacheMiss = errors.New("cache missing")
)

type Options struct {
	FlagPrefix string
}

func (o Options) GetEnvPrefix() string {
	return strings.ReplaceAll(strings.ToUpper(o.FlagPrefix), "-", "_")
}

const (
	FlagRedis              = "redis"
	FlagRedisDB            = "redisdb"
	FlagRedisCompress      = "redis-compress"
	FlagDefaultCacheExpire = "default-cache-expire"
)

func AddRedisCacheFlagsToCmd(cmd *cobra.Command, opt Options) func() (*Cache, error) {
	var (
		redisAddr          string
		redisDB            int
		redisCompress      string
		defaultCacheExpire time.Duration
	)
	cmd.Flags().StringVar(&redisAddr, opt.FlagPrefix+FlagRedis, env.StringFromEnv(opt.GetEnvPrefix()+"REDIS_SERVER", ""), "")
	redisAddrFn := getFlagVal(cmd, opt, FlagRedis, cmd.Flags().GetString)

	cmd.Flags().IntVar(&redisDB, opt.FlagPrefix+FlagRedisDB, env.ParseNumFromEnv(opt.GetEnvPrefix()+"REDIS_DB", 0, 0, math.MaxInt), "")
	redisDBFn := getFlagVal(cmd, opt, FlagRedisDB, cmd.Flags().GetInt)

	cmd.Flags().StringVar(&redisCompress, opt.FlagPrefix+FlagRedisCompress, env.StringFromEnv(opt.GetEnvPrefix()+"REDIS_COMPRESS", ""), "gzip or left empty")
	redisCompressFn := getFlagVal(cmd, opt, FlagRedisCompress, cmd.Flags().GetString)

	cmd.Flags().DurationVar(&defaultCacheExpire, opt.FlagPrefix+FlagDefaultCacheExpire, env.ParseDurationFromEnv(opt.GetEnvPrefix()+"DEFAULT_CACHE_EXPIRE", 24*time.Hour, 0, math.MaxInt64), "cache expiration")
	defaultCacheExpireFn := getFlagVal(cmd, opt, FlagDefaultCacheExpire, cmd.Flags().GetDuration)

	return func() (*Cache, error) {
		password := os.Getenv("REDIS_USERNAME")
		username := os.Getenv("REDIS_PASSWORD")
		if opt.FlagPrefix != "" {
			if val := os.Getenv(opt.GetEnvPrefix() + "REDIS_USERNAME"); val != "" {
				username = val
			}
			if val := os.Getenv(opt.GetEnvPrefix() + "REDIS_PASSWORD"); val != "" {
				password = val
			}
		}
		client, err := buildRedisClient(
			redisAddrFn(),
			username,
			password,
			redisDBFn(),
		)
		if err != nil {
			return nil, err
		}
		expiration, compress := defaultCacheExpireFn(), redisCompressFn()
		return &Cache{client: NewRedisCache(client, expiration, compress)}, nil
	}
}

func buildRedisClient(addr, username, password string, db int) (redis.UniversalClient, error) {
	client := redis.NewUniversalClient(&redis.UniversalOptions{
		Addrs:    strings.Split(addr, ","),
		DB:       db,
		Username: username,
		Password: password,
	})
	if err := client.Ping(context.TODO()).Err(); err != nil {
		return nil, err
	}
	return client, nil
}

func getFlagVal[T any](cmd *cobra.Command, opt Options, name string, getVal func(name string) (T, error)) func() T {
	return func() T {
		var (
			res T
			err error
		)
		if opt.FlagPrefix != "" && cmd.Flags().Changed(opt.FlagPrefix+name) {
			res, err = getVal(opt.FlagPrefix + name)
		} else {
			res, err = getVal(name)
		}
		if err != nil {
			panic(err)
		}
		return res
	}
}

type Cache struct {
	client CacheClient
}

func (c *Cache) GetClient() CacheClient {
	return c.client
}

func (c *Cache) SetClient(client CacheClient) {
	c.client = client
}

func (c *Cache) RenameItem(oldKey string, newKey string, expiration time.Duration) error {
	return c.client.Rename(oldKey, newKey, expiration)
}

func (c *Cache) SetItem(key string, item interface{}, expiration time.Duration, delete bool) error {
	if delete {
		return c.client.Delete(key)
	} else {
		if item == nil {
			return fmt.Errorf("cannot set item to nil for key %s", key)
		}
		return c.client.Set(&Item{Value: item, Key: key, Duration: expiration})
	}
}

func (c *Cache) GetItem(key string, item interface{}) error {
	if item == nil {
		return fmt.Errorf("cannot get item into a nil for key %s", key)
	}
	return c.client.Get(key, item)
}

func (c *Cache) OnUpdated(ctx context.Context, key string, callback func() error) error {
	return c.client.OnUpdated(ctx, key, callback)
}

func (c *Cache) NotifyUpdated(key string) error {
	return c.client.NotifyUpdated(key)
}
