package cache

import (
	"bytes"
	"compress/gzip"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"time"

	rediscache "github.com/go-redis/cache/v9"
	"github.com/redis/go-redis/v9"
)

func NewRedisCache(client redis.UniversalClient, expiration time.Duration, compress string) CacheClient {
	return &redisCache{
		client:            client,
		redisCompressType: compress,
		expiration:        expiration,
		cache: rediscache.New(&rediscache.Options{
			Redis: client,
		}),
	}
}

var _ CacheClient = &redisCache{}

type redisCache struct {
	client            redis.UniversalClient
	cache             *rediscache.Cache
	redisCompressType string
	expiration        time.Duration
}

func (rc *redisCache) Set(item *Item) error {
	expiration := item.Duration
	if expiration == 0 {
		expiration = rc.expiration
	}
	val, err := rc.serialize(item.Value)
	if err != nil {
		return err
	}
	return rc.cache.Set(&rediscache.Item{
		Key:   rc.getKey(item.Key),
		Value: val,
		TTL:   expiration,
	})
}

func (rc *redisCache) Rename(oldKey string, newKey string, expiration time.Duration) error {
	return nil
}

func (rc *redisCache) Get(key string, obj interface{}) error {
	var data []byte
	if err := rc.cache.Get(context.TODO(), rc.getKey(key), &data); err != nil {
		return err
	}
	if err := rc.deserialize(data, obj); err != nil {
		return err
	}
	return nil
}

func (rc *redisCache) Delete(key string) error {
	return rc.cache.Delete(context.TODO(), rc.getKey(key))
}

func (rc *redisCache) OnUpdated(ctx context.Context, key string, callback func() error) error {
	return nil
}

func (rc *redisCache) NotifyUpdated(key string) error {
	return nil
}

func (r *redisCache) getKey(key string) string {
	if r.redisCompressType == "gzip" {
		return key + ".gz"
	}
	return key
}

func (r *redisCache) serialize(obj interface{}) ([]byte, error) {
	buf := bytes.NewBuffer([]byte{})
	var w io.Writer = buf
	if r.redisCompressType == "gzip" {
		w = gzip.NewWriter(buf)
	}
	encoder := json.NewEncoder(w)

	if err := encoder.Encode(obj); err != nil {
		return nil, err
	}
	if flusher, ok := w.(interface{ Flush() error }); ok {
		if err := flusher.Flush(); err != nil {
			return nil, err
		}
	}
	return buf.Bytes(), nil
}

func (r *redisCache) deserialize(data []byte, obj interface{}) error {
	buf := bytes.NewReader(data)
	var reader io.Reader = buf
	if r.redisCompressType == "gzip" {
		if gzipReader, err := gzip.NewReader(buf); err != nil {
			return err
		} else {
			reader = gzipReader
		}
	}
	if err := json.NewDecoder(reader).Decode(obj); err != nil {
		return fmt.Errorf("failed to decode cached data: %w", err)
	}
	return nil
}
